﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

namespace OGSNET
{
    using OGSNET.Plugin;

    public partial class SettingForm : Form
    {
        protected PluginControl PluginControl; /**< プラグイン管理オブジェクト */
        protected AccountDataControl AccountDataControl; /**< アカウント管理オブジェクト */

        /** \brief 運営プラグインのデータ
         * 
         * StringPair.First プラグインID
         * StringPair.Second プラグイン名
         */
        protected List<StringPair> OperationPlugin;

        /** \brief ゲームプラグインのデータ
         * 
         * StringPair.First プラグインID
         * StringPair.Second プラグイン名
         */
        protected List<StringPair> GamePlugin;

        /**
         * \brief フォームの初期化
         * \arg \c pluginControl プラグイン管理オブジェクト
         * \arg \c accountDataControl アカウント管理オブジェクト
         */
        public SettingForm(PluginControl pluginControl, AccountDataControl accountDataControl)
        {
            InitializeComponent();

            // インスタンス変数の代入
            this.PluginControl = pluginControl;
            this.AccountDataControl = accountDataControl;

            // 一覧の作成
            this.CreatePluginList();
            this.ChangePluginClass(this.OperationPlugin);
        }

        /** \brief 内部処理で使うプラグイン一覧を作成する */
        protected void CreatePluginList()
        {
            // 初期化
            this.OperationPlugin = new List<StringPair>();
            this.GamePlugin      = new List<StringPair>();

            var plugins = this.PluginControl.GetPlugins();

            foreach (var plugin in plugins)
            {
                // (プラグインId, プラグイン名) のペア
                var pair = new StringPair(plugin.Metadata.PluginId, plugin.Value.GetPluginInfo().PluginName);

                // 運営プラグインの場合
                if (plugin.Value is IOperationPlugin)
                {
                    this.OperationPlugin.Add(pair);
                }

                // ゲームプラグインの場合
                else if (plugin.Value is IGamePlugin)
                {
                    this.GamePlugin.Add(pair);
                }
            }

            // 並び替える
            this.OperationPlugin.Sort();
            this.GamePlugin.Sort();
        }

        /**
         * \brief プラグインの分類の変更処理
         * \arg \c pairs 表示するプラグインのペア一覧
         */
        private void ChangePluginClass(List<StringPair> pairs)
        {
            lock (this.cmbPlugin)
            {
                // 一覧をクリア
                this.cmbPlugin.Items.Clear();

                foreach (var pair in pairs)
                {
                    if (pair.ToString() != null)
                    {
                        this.cmbPlugin.Items.Add(pair);
                    }
                }

                if (this.cmbPlugin.Items.Count > 0)
                {
                    this.cmbPlugin.SelectedIndex = 0;
                }
            }

            // 0個以上あれば、プラグインの変更処理を実行
            if (this.cmbPlugin.Items.Count > 0)
            {
                this.ChangePlugin();
            }
        }

        /** \brief プラグインの変更処理 */
        private void ChangePlugin()
        {
            // プラグインの個数が0の場合
            if(this.cmbPlugin.Items.Count == 0) return;

            // 一覧に追加ボタンを無効化
            this.btnDeleteList.Enabled = false;

            // 追加ボタンを有効化
            this.btnAddList.Enabled = true;

            // 削除ボタンを無効化
            this.btnRemove.Enabled = false;

            // 現在選択中のプラグイン
            var pair = (StringPair)this.cmbPlugin.SelectedItem;

            // 現在選択中のプラグインのアカウントを列挙
            lock (this.cmbAccount)
            {
                // アカウント一覧をクリア
                this.cmbAccount.Items.Clear();

                foreach (var account in this.AccountDataControl.Accounts)
                {
                    if (account.PluginId == pair.First)
                    {
                        // アカウントが有効な場合 (ダミーアカウントでない)
                        if (account.IsEnable)
                        {
                            this.cmbAccount.Items.Add(account);
                        }

                        // ダミーアカウントの場合
                        else
                        {
                            this.btnDeleteList.Enabled = true;
                            this.btnAddList.Enabled    = false;
                        }
                    }
                }

                // アカウント数が0以上な場合
                if (this.cmbAccount.Items.Count > 0)
                {
                    // 削除ボタンを有効化
                    this.btnRemove.Enabled = true;

                    // 一番上を選択
                    this.cmbAccount.SelectedIndex = 0;
                }
            }

            // プラグインを取得
            var plugin = this.PluginControl.GetPlugin(pair.First);

            // 運営プラグインの場合は、一覧へ追加ボタンを無効化
            if (plugin.Value is IOperationPlugin)
            {
                this.btnAddList.Enabled = false;
            }
            
            // プラグインの情報を表示させる
            var pluginInfo = plugin.Value.GetPluginInfo();

            var comment = pluginInfo.Comment + "\r\n\r\n";
            comment += "バージョン: " + pluginInfo.Version + "\r\n";
            comment += "制作者: " + pluginInfo.AuthorName + "\r\n";

            if (plugin.Value is IGamePlugin)
            {
                var gamePlugin = plugin.Value as IGamePlugin;
                var gameInfo = gamePlugin.GetGameInfo();

                comment += "運営: " + gameInfo.OperationName;
            }

            this.lbComment.Text = comment;
        }

        /** \brief アカウントの追加処理 */
        private void AddAccount(AccountData account)
        {
            // 既に存在する
            if (this.AccountDataControl.Accounts.Contains(account))
            {
                MessageBox.Show("アカウントが既に存在します。パスワードを変更する場合は既にあるものを先に削除してください。");
            }
            else
            {
                MessageBox.Show("アカウントを追加しました。");

                // アカウント一覧の更新
                this.AccountDataControl.Accounts.Add(account);
                this.ChangePlugin();
            }
        }

        /**
         * \brief 入力されてるアカウントを追加する
         * \arg \c isList ゲーム一覧として追加するかの真偽値
         */
        private void AddInputAccount(bool isList = false)
        {
            if (this.cmbPlugin.Items.Count == 0) return;

            // 現在選択中のプラグイン
            var pair = (StringPair)this.cmbPlugin.SelectedItem;

            // ユーザー名とパスワード
            var userName = this.txtUserName.Text;
            var password = this.txtPassword.Text;

            if (isList)
            {
                // ダミーアカウントを追加
                this.AddAccount(new AccountData(pair.First));
            }

            // ユーザー名とパスワードが入力されている場合
            else if (userName.Length > 0 && password.Length > 0)
            {
                this.AddAccount(new AccountData(pair.First, userName, password));

                this.txtUserName.Clear();
                this.txtPassword.Clear();
            }
            else
            {
                MessageBox.Show("データを正しく入力してください。");
            }
        }

        /**
         * \brief 入力されているか調べる
         * 
         * 入力欄が変更されたときに呼ばれる
         *
         */
        private void CheckInput()
        {
            if (this.txtUserName.Text.Length > 0 && this.txtPassword.Text.Length > 0)
            {
                this.btnAdd.Enabled = true;
            }
            else
            {
                this.btnAdd.Enabled = false;
            }
        }


        /** \brief アカウントを削除する */
        private void RemoveAccountSelected()
        {
            if (this.cmbAccount.Items.Count == 0) return;

            AccountData account;

            lock (this.cmbAccount)
            {
                account = (AccountData)this.cmbAccount.SelectedItem;

                this.cmbAccount.Items.Remove(account);
            }

            this.AccountDataControl.Accounts.Remove(account);
            this.ChangePlugin();
        }

        /** \brief ゲーム一覧として追加されたアカウントを削除する */
        private void RemoveAccountList()
        {
            if (this.cmbPlugin.Items.Count == 0) return;

            // 現在選択されているプラグイン
            var pair = (StringPair)this.cmbPlugin.SelectedItem;

            // ゲームプラグインかどうか
            if (this.GamePlugin.Contains(pair))
            {
                // 削除する
                AccountData account = new AccountData(pair.First);
                this.AccountDataControl.Accounts.Remove(account);

                this.ChangePlugin();
            }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            lock (this)
            {
                this.DialogResult = DialogResult.OK;
                this.Close();
            }
        }

        private void rdGame_CheckedChanged(object sender, EventArgs e)
        {
            this.ChangePluginClass(this.GamePlugin);
        }

        private void rdOperation_CheckedChanged(object sender, EventArgs e)
        {
            this.ChangePluginClass(this.OperationPlugin);
        }

        private void txtPassword_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                this.AddInputAccount();   
            }
        }

        private void btnAdd_Click(object sender, EventArgs e)
        {
            this.AddInputAccount();
        }

        private void cmbPlugin_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.ChangePlugin();
        }

        private void txtUserName_TextChanged(object sender, EventArgs e)
        {
            this.CheckInput();
        }

        private void txtPassword_TextChanged(object sender, EventArgs e)
        {
            this.CheckInput();
        }

        private void btnAddList_Click(object sender, EventArgs e)
        {
            this.AddInputAccount(true);
        }

        private void btnDeleteList_Click(object sender, EventArgs e)
        {
            this.RemoveAccountList();   
        }

        private void btnRemove_Click(object sender, EventArgs e)
        {
            this.RemoveAccountSelected();
        }
    }

}
